require('dotenv').config();
const puppeteer = require('puppeteer');
const fs = require('fs');
const path = require('path');

// Get store configuration from command line arguments OR environment
const args = process.argv.slice(2);
const STORE_LOCATION = args[0] || process.env.STORE_LOCATION || 'plum';
const STORE_NUMBER = args[1] || process.env.STORE_NUMBER || '19556';
const STORE_NAME = args[2] || process.env.STORE_NAME || 'Plum';
// ALWAYS use headless for the actual ParBrink scraping
const HEADLESS_MODE = true;

console.log('\n╔══════════════════════════════════════════════════════════╗');
console.log('║                                                          ║');
console.log('║          D&K DAIRY QUEEN - NIGHTLY REPORT                ║');
console.log('║                                                          ║');
console.log('╚══════════════════════════════════════════════════════════╝\n');

console.log(`Store: ${STORE_NAME}`);
console.log(`Store Number: ${STORE_NUMBER}`);
console.log(`Time: ${new Date().toLocaleString('en-US', { timeZone: 'America/New_York' })}`);
console.log(`\nStarting report fetch...\n`);

async function fetchReport() {
  let browser = null;

  try {
    const credentials = {
      username: process.env.PARBRINK_USERNAME,
      password: process.env.PARBRINK_PASSWORD,
      systemUrl: process.env.PARBRINK_URL
    };

    console.log('Logging into ParBrink...');

    // Launch browser in HEADLESS mode (user sees loading.html instead)
    browser = await puppeteer.launch({
      headless: HEADLESS_MODE,
      args: ['--no-sandbox', '--disable-setuid-sandbox', '--disable-features=site-per-process'],
      defaultViewport: { width: 1280, height: 720 }
    });

    const page = await browser.newPage();

    // Set longer default timeout for navigation (helps with Indiana)
    page.setDefaultNavigationTimeout(30000);
    page.setDefaultTimeout(30000);

    // Navigate to login page
    console.log('   Loading login page...');
    await page.goto(credentials.systemUrl, { waitUntil: 'domcontentloaded', timeout: 30000 });

    // Wait for form
    await page.waitForSelector('form', { timeout: 10000 });

    // Fill username
    await page.evaluate((username) => {
      const usernameFields = Array.from(document.querySelectorAll('input[name="Username"], input#Username, input[type="text"]'));
      const visibleField = usernameFields.find(field => {
        const rect = field.getBoundingClientRect();
        return rect.width > 0 && rect.height > 0;
      });
      if (visibleField) {
        visibleField.value = username;
        visibleField.dispatchEvent(new Event('input', { bubbles: true }));
        visibleField.dispatchEvent(new Event('change', { bubbles: true }));
      }
    }, credentials.username);

    // Fill password
    await page.evaluate((password) => {
      const passwordFields = Array.from(document.querySelectorAll('input[name="Password"], input#Password, input[type="password"]'));
      const visibleField = passwordFields.find(field => {
        const rect = field.getBoundingClientRect();
        return rect.width > 0 && rect.height > 0;
      });
      if (visibleField) {
        visibleField.value = password;
        visibleField.dispatchEvent(new Event('input', { bubbles: true }));
        visibleField.dispatchEvent(new Event('change', { bubbles: true }));
      }
    }, credentials.password);

    await new Promise(resolve => setTimeout(resolve, 500));

    // Click login
    console.log('   Logging in...');
    const navigationPromise = page.waitForNavigation({
      waitUntil: 'domcontentloaded',
      timeout: 10000
    }).catch(() => null);

    await page.evaluate(() => {
      const button = document.querySelector('button[type="submit"]');
      if (button) button.click();
    });

    await navigationPromise;
    await new Promise(resolve => setTimeout(resolve, 2000));

    // Check if login was successful
    const currentUrl = page.url();
    const loginFailed = await page.evaluate(() => {
      // Check for error messages
      const errorText = document.body.textContent;
      return errorText.includes('Invalid username or password') ||
             errorText.includes('Login failed') ||
             errorText.includes('Incorrect username') ||
             errorText.includes('Incorrect password');
    });

    if (loginFailed || currentUrl.includes('login')) {
      console.error('\n❌ LOGIN FAILED - Invalid username or password!');
      console.error('═══════════════════════════════════════════════════════════');
      console.error('Please check your .env file and update the credentials:');
      console.error('  1. Open the .env file in this folder');
      console.error('  2. Update PARBRINK_USERNAME and PARBRINK_PASSWORD');
      console.error('  3. Run the report again');
      console.error('═══════════════════════════════════════════════════════════\n');
      if (browser) await browser.close();
      process.exit(2); // Exit code 2 for authentication error
    }

    console.log('   Login successful!\n');

    // Navigate to Sales Summary
    console.log('Navigating to Sales Summary...');
    const salesSummaryUrl = 'https://admin7.parpos.com/Reports/Report/SalesSummary/';

    // Use networkidle2 for better stability (especially for Indiana)
    await page.goto(salesSummaryUrl, { waitUntil: 'networkidle2', timeout: 30000 });

    // Extra wait for page to fully stabilize
    await new Promise(resolve => setTimeout(resolve, 3000));

    // Click Locations dropdown
    console.log('   Selecting location...');
    await page.waitForSelector('body', { timeout: 10000 });

    // Extra wait for Indiana store to ensure page is ready
    await new Promise(resolve => setTimeout(resolve, 2000));

    try {
      await page.evaluate(() => {
        const allElements = Array.from(document.querySelectorAll('button, div, span, label, a'));
        const locationsElement = allElements.find(el => {
          const text = el.textContent?.trim();
          return text === 'Locations' || text?.includes('Locations');
        });
        if (locationsElement) {
          locationsElement.click();
        }
      });
    } catch (e) {
      console.log('   Warning: Error clicking Locations, retrying...');
      await new Promise(resolve => setTimeout(resolve, 1000));
      await page.evaluate(() => {
        const allElements = Array.from(document.querySelectorAll('button, div, span, label, a'));
        const locationsElement = allElements.find(el => {
          const text = el.textContent?.trim();
          return text === 'Locations' || text?.includes('Locations');
        });
        if (locationsElement) {
          locationsElement.click();
        }
      });
    }

    await new Promise(resolve => setTimeout(resolve, 2000));

    // Select store from dropdown with retry and navigation handling
    const storeNavigationPromise = new Promise((resolve) => {
      const navListener = () => {
        console.log('   Page reloaded after store selection');
        resolve();
      };
      page.once('framenavigated', navListener);
      setTimeout(() => {
        page.off('framenavigated', navListener);
        resolve();
      }, 8000);
    });

    try {
      await page.evaluate((storeNum) => {
        const dropdowns = Array.from(document.querySelectorAll('select, .dx-dropdowneditor-input-wrapper, .dx-texteditor-input'));

        for (const dropdown of dropdowns) {
          if (dropdown.tagName === 'SELECT') {
            const options = Array.from(dropdown.querySelectorAll('option'));
            const matchingOption = options.find(opt => opt.textContent.includes(storeNum));
            if (matchingOption) {
              dropdown.value = matchingOption.value;
              dropdown.dispatchEvent(new Event('change', { bubbles: true }));
              return;
            }
          }
          dropdown.click();
        }

        setTimeout(() => {
          const allOptions = Array.from(document.querySelectorAll('div, span, li, .dx-item, .dx-list-item'));
          const matchingOption = allOptions.find(el => {
            const text = el.textContent || '';
            return text.includes(storeNum);
          });

          if (matchingOption) {
            matchingOption.click();
          }
        }, 500);
      }, STORE_NUMBER);
    } catch (e) {
      console.log('   Store selection may have triggered navigation');
    }

    // Wait for any navigation
    await storeNavigationPromise;

    await new Promise(resolve => setTimeout(resolve, 3000));

    // Set date to today (EST)
    console.log('   Setting date to today...');

    // Set up navigation listener in case date change triggers page reload (Indiana issue)
    const dateNavigationPromise = new Promise((resolve) => {
      const navigationListener = () => {
        console.log('   Page reloaded after date change (this is normal for Indiana)');
        resolve();
      };
      page.once('framenavigated', navigationListener);
      // Auto-resolve after timeout if no navigation
      setTimeout(() => {
        page.off('framenavigated', navigationListener);
        resolve();
      }, 10000);
    });

    try {
      await page.evaluate(() => {
        const today = new Date(new Date().toLocaleString('en-US', { timeZone: 'America/New_York' }));

        const month = String(today.getMonth() + 1).padStart(2, '0');
        const day = String(today.getDate()).padStart(2, '0');
        const year = today.getFullYear();
        const dateString = `${month}/${day}/${year}`;

        const dateInputs = Array.from(document.querySelectorAll('input[type="text"], input[type="date"], .dx-texteditor-input'));

        for (const input of dateInputs) {
          const label = input.placeholder || input.getAttribute('aria-label') || '';
          const parentText = input.parentElement?.textContent || '';

          if (label.toLowerCase().includes('from') || label.toLowerCase().includes('start') ||
              parentText.toLowerCase().includes('from') || parentText.toLowerCase().includes('start')) {
            input.value = dateString;
            input.dispatchEvent(new Event('input', { bubbles: true }));
            input.dispatchEvent(new Event('change', { bubbles: true }));
          }

          if (label.toLowerCase().includes('to') || label.toLowerCase().includes('end') ||
              parentText.toLowerCase().includes('to') || parentText.toLowerCase().includes('end')) {
            input.value = dateString;
            input.dispatchEvent(new Event('input', { bubbles: true }));
            input.dispatchEvent(new Event('change', { bubbles: true }));
          }
        }
      });
    } catch (e) {
      console.log('   Date setting triggered navigation (Indiana store)');
    }

    // Wait for any navigation that might have been triggered
    await dateNavigationPromise;

    // Extra wait for page to stabilize after date change
    await new Promise(resolve => setTimeout(resolve, 3000));

    // Click View Report
    console.log('   Generating report...');

    // Add longer wait before clicking to ensure page is ready
    await new Promise(resolve => setTimeout(resolve, 3000));

    // Set up navigation listener BEFORE clicking button
    const viewReportNavigationPromise = new Promise((resolve) => {
      page.once('framenavigated', () => {
        console.log('   Page navigated successfully');
        resolve();
      });
      // Auto-resolve after timeout if no navigation occurs
      setTimeout(resolve, 20000);
    });

    try {
      await page.evaluate(() => {
        const buttons = Array.from(document.querySelectorAll('button, input[type="submit"]'));
        const viewReportButton = buttons.find(btn => {
          const text = btn.textContent?.trim() || '';
          const value = btn.value || '';
          return text === 'View Report' || value === 'View Report';
        });
        if (viewReportButton) {
          viewReportButton.click();
        }
      });
    } catch (e) {
      console.log('   Warning: Error clicking View Report button');
    }

    // Wait for report to load
    console.log('   Waiting for report to load (this takes 30-60 seconds)...');

    // Wait for navigation or timeout
    await viewReportNavigationPromise;

    // Long wait for page to fully stabilize
    await new Promise(resolve => setTimeout(resolve, 10000));

    // Wait for loading panel to disappear with longer timeout
    try {
      await page.waitForFunction(
        () => {
          const loadingPanel = document.querySelector('#reportViewer1_LP, .dxLoadingPanel');
          if (!loadingPanel) return true;
          const style = window.getComputedStyle(loadingPanel);
          return style.display === 'none';
        },
        { timeout: 90000 }
      );
    } catch (e) {
      console.log('   Loading panel timeout - continuing anyway...');
    }

    // Extra long wait to ensure page is fully stable (especially for Indiana)
    await new Promise(resolve => setTimeout(resolve, 5000));

    console.log('   Report loaded!\n');

    // Extract data from ALL 3 pages
    console.log('Extracting sales data from all pages...\n');
    console.log('   Page 1...');

    // Get page 1 text
    let page1Text = '';
    try {
      page1Text = await page.evaluate(() => {
        let iframe = document.querySelector('iframe') || document.querySelector('#reportViewer1_ReportArea iframe');
        if (iframe && iframe.contentDocument) {
          return iframe.contentDocument.body.innerText || '';
        }
        return document.body.innerText || '';
      });
    } catch (e) {
      console.log('   Warning: Error extracting page 1 text, retrying...');
      await new Promise(resolve => setTimeout(resolve, 2000));
      try {
        page1Text = await page.evaluate(() => {
          let iframe = document.querySelector('iframe') || document.querySelector('#reportViewer1_ReportArea iframe');
          if (iframe && iframe.contentDocument) {
            return iframe.contentDocument.body.innerText || '';
          }
          return document.body.innerText || '';
        });
      } catch (e2) {
        console.log('   Warning: Could not extract page 1 text');
      }
    }

    // Go to page 2
    console.log('   Navigating to Page 2...');
    try {
      await page.evaluate(() => {
        const pageInputs = Array.from(document.querySelectorAll('input[type="text"], input[type="number"]'));
        const pageInput = pageInputs.find(inp => {
          const value = inp.value || '';
          return value === '1' || value.match(/^\d+$/);
        });

        if (pageInput) {
          pageInput.value = '2';
          pageInput.dispatchEvent(new Event('input', { bubbles: true }));
          pageInput.dispatchEvent(new Event('change', { bubbles: true }));
          pageInput.dispatchEvent(new KeyboardEvent('keydown', { key: 'Enter', keyCode: 13, bubbles: true }));
        }
      });
    } catch (e) {
      console.log('   Warning: Could not navigate to page 2, continuing...');
    }
    await new Promise(resolve => setTimeout(resolve, 5000));

    let page2Text = '';
    try {
      page2Text = await page.evaluate(() => {
        let iframe = document.querySelector('iframe') || document.querySelector('#reportViewer1_ReportArea iframe');
        if (iframe && iframe.contentDocument) {
          return iframe.contentDocument.body.innerText || '';
        }
        return document.body.innerText || '';
      });
    } catch (e) {
      console.log('   Warning: Could not extract page 2 text');
    }

    // Go to page 3
    console.log('   Navigating to Page 3...');
    try {
      await page.evaluate(() => {
        const pageInputs = Array.from(document.querySelectorAll('input[type="text"], input[type="number"]'));
        const pageInput = pageInputs.find(inp => {
          const value = inp.value || '';
          return value === '2' || value.match(/^\d+$/);
        });

        if (pageInput) {
          pageInput.value = '3';
          pageInput.dispatchEvent(new Event('input', { bubbles: true }));
          pageInput.dispatchEvent(new Event('change', { bubbles: true }));
          pageInput.dispatchEvent(new KeyboardEvent('keydown', { key: 'Enter', keyCode: 13, bubbles: true }));
        }
      });
    } catch (e) {
      console.log('   Warning: Could not navigate to page 3, continuing...');
    }
    await new Promise(resolve => setTimeout(resolve, 5000));

    let page3Text = '';
    try {
      page3Text = await page.evaluate(() => {
        let iframe = document.querySelector('iframe') || document.querySelector('#reportViewer1_ReportArea iframe');
        if (iframe && iframe.contentDocument) {
          return iframe.contentDocument.body.innerText || '';
        }
        return document.body.innerText || '';
      });
    } catch (e) {
      console.log('   Warning: Could not extract page 3 text');
    }

    console.log('   All pages extracted!\n');

    // Combine all 3 pages
    const combinedText = page1Text + '\n\n--- PAGE 2 ---\n\n' + page2Text + '\n\n--- PAGE 3 ---\n\n' + page3Text;

    const extractionResult = await page.evaluate((pageText) => {
      // IMPORTANT: pageText contains ALL 3 pages combined
      // Fields can appear on different pages depending on data volume
      // So we search the ENTIRE combined text for ALL fields
      window.debugPageText = pageText;

      // Initialize all fields
      let totalMedia = 0;
      let totalCash = 0;
      let creditCards = 0;
      let food = 0;
      let salesTax = 0;
      let customerCount = 0;
      let avgGuestCheck = 0;
      let nonTaxableSales = 0;
      let cakesCount = 0;
      let cupcakesCount = 0;

      // Extract Total Media (from Tenders section - can be on any page)
      let match = pageText.match(/Tenders[\s\S]*?Total[\s\t]+\d+[\s\t]+\$?([\d,]+\.\d{2})/i);
      if (match) {
        totalMedia = parseFloat(match[1].replace(/,/g, ''));
      }

      if (totalMedia === 0) {
        match = pageText.match(/Total\s+\d+\s+\$?[\d,]+\.\d{2}\s+\$?[\d,]+\.\d{2}\s+\$?([\d,]+\.\d{2})/i);
        if (match) {
          totalMedia = parseFloat(match[1].replace(/,/g, ''));
        }
      }

      if (totalMedia === 0) {
        const tendersMatch = pageText.match(/Tenders[\s\S]{50,500}Total[\s\S]{1,100}/i);
        if (tendersMatch) {
          const amounts = tendersMatch[0].match(/\$([\d,]+\.\d{2})/g);
          if (amounts && amounts.length > 0) {
            const lastAmount = amounts[amounts.length - 1];
            totalMedia = parseFloat(lastAmount.replace(/[$,]/g, ''));
          }
        }
      }

      // Extract Total Cash (can be on any page)
      match = pageText.match(/Total\s+Cash[:\s]+\$?([\d,]+\.\d{2})/i);
      if (match) {
        totalCash = parseFloat(match[1].replace(/,/g, ''));
      }

      // Extract Credit Cards (from Non-Cash Payments - can be on any page)
      match = pageText.match(/Non[-\s]?Cash\s+Payments[:\s]+\$?([\d,]+\.\d{2})/i);
      if (match) {
        creditCards = parseFloat(match[1].replace(/,/g, ''));
      }

      if (creditCards === 0) {
        match = pageText.match(/Non\s+Cash\s+Payments[\s\S]{0,50}?\$?([\d,]+\.\d{2})/i);
        if (match) {
          creditCards = parseFloat(match[1].replace(/,/g, ''));
        }
      }

      // Extract Food (from Revenue Centers - can be on any page)
      match = pageText.match(/Revenue\s+Centers[\s\S]{0,200}?Food[:\s]+\$?([\d,]+\.\d{2})/i);
      if (match) {
        food = parseFloat(match[1].replace(/,/g, ''));
      }

      if (food === 0) {
        const revenueCentersMatch = pageText.match(/Revenue\s+Centers[\s\S]{0,500}/i);
        if (revenueCentersMatch) {
          const foodMatch = revenueCentersMatch[0].match(/Food[\s\S]{0,50}?\$?([\d,]+\.\d{2})/i);
          if (foodMatch) {
            food = parseFloat(foodMatch[1].replace(/,/g, ''));
          }
        }
      }

      if (food === 0) {
        match = pageText.match(/Food\s+\$?([\d,]+\.\d{2})/i);
        if (match) {
          food = parseFloat(match[1].replace(/,/g, ''));
        }
      }

      // Extract Sales Tax (can be on any page - search entire combined text)
      // Pattern 1: "+ Tax" format (Murrysville/Plum format)
      // Example: "+ Tax		$73.35"
      match = pageText.match(/\+\s+Tax[\s\t]+\$?([\d,]+\.\d{2})/i);
      if (match) {
        salesTax = parseFloat(match[1].replace(/,/g, ''));
      }

      // Pattern 2: Table format - "Sales Tax" + tabs + quantity + tabs + dollar amount
      // Example: "Sales Tax		252				$70.37		100.00%"
      if (salesTax === 0) {
        match = pageText.match(/Sales\s+Tax[\s\t]+\d+[\s\t]+\$?([\d,]+\.\d{2})/i);
        if (match) {
          salesTax = parseFloat(match[1].replace(/,/g, ''));
        }
      }

      // Pattern 3: "Sales Tax" with colon or spaces directly to amount
      if (salesTax === 0) {
        match = pageText.match(/Sales\s+Tax[\s\t:]+\$?([\d,]+\.\d{2})/i);
        if (match) {
          salesTax = parseFloat(match[1].replace(/,/g, ''));
        }
      }

      // Pattern 4: "Sales Tax Total"
      if (salesTax === 0) {
        match = pageText.match(/Sales\s+Tax\s+Total[\s\t:]+\$?([\d,]+\.\d{2})/i);
        if (match) {
          salesTax = parseFloat(match[1].replace(/,/g, ''));
        }
      }

      // Pattern 5: Table format - "Sales Tax" anywhere on line, then dollar amount
      if (salesTax === 0) {
        match = pageText.match(/Sales\s+Tax[^\n]{0,100}?\$?([\d,]+\.\d{2})/i);
        if (match) {
          salesTax = parseFloat(match[1].replace(/,/g, ''));
        }
      }

      // Pattern 6: Just "Tax" with dollar amount (without + sign)
      if (salesTax === 0) {
        match = pageText.match(/(?:^|\n)\s*Tax[\s\t:]+\$?([\d,]+\.\d{2})/im);
        if (match) {
          salesTax = parseFloat(match[1].replace(/,/g, ''));
        }
      }

      // Pattern 7: Look specifically on page 3 for any tax-related field
      if (salesTax === 0) {
        const page3Match = pageText.match(/--- PAGE 3 ---[\s\S]+$/);
        if (page3Match) {
          const taxOnPage3 = page3Match[0].match(/(?:Sales\s+)?Tax[\s\t:]+\$?([\d,]+\.\d{2})/i);
          if (taxOnPage3) {
            salesTax = parseFloat(taxOnPage3[1].replace(/,/g, ''));
          }
        }
      }

      // Extract Customer Count (from Guest Count - can be on any page)
      const guestMatch = pageText.match(/Guest\s+Count\s*:[\s\S]{1,250}?Net\s+Sales\s+\$?[\d,]+\.?\d*\s+(\d{1,3})\s/i);
      if (guestMatch) {
        customerCount = parseInt(guestMatch[1]);
      }

      if (customerCount === 0) {
        const lines = pageText.split('\n');
        for (let i = 0; i < lines.length; i++) {
          if (lines[i].match(/Guest\s+Count/i)) {
            for (let j = i + 1; j < Math.min(i + 5, lines.length); j++) {
              const numMatch = lines[j].match(/(?<!\$)\s+(\d{1,3})\s*$/);
              if (numMatch) {
                const num = parseInt(numMatch[1]);
                if (num > 0 && num < 500) {
                  customerCount = num;
                  break;
                }
              }
            }
            if (customerCount > 0) break;
          }
        }
      }

      // Extract Average Guest Check (from Order Average - can be on any page)
      match = pageText.match(/Order\s+Average[:\s]+\$?([\d,]+\.\d{2})/i);
      if (match) {
        avgGuestCheck = parseFloat(match[1].replace(/,/g, ''));
      }

      // Extract Non-Taxable Item Sales (can be on any page)
      match = pageText.match(/Non[-\s]?Taxable\s+(?:Item\s+)?Sales[:\s]+\$?([\d,]+\.\d{2})/i);
      if (match) {
        nonTaxableSales = parseFloat(match[1].replace(/,/g, ''));
      }

      // Extract Cake Count (can be on any page)
      const cake8Match = pageText.match(/8["\s]*(?:INCH|IN)?\s*(?:ICE\s+CREAM\s+)?CAKE[S]?[\s\S]{0,50}?(\d+)/i);
      if (cake8Match) {
        cakesCount += parseInt(cake8Match[1]);
      }

      const cake10Match = pageText.match(/10["\s]*(?:INCH|IN)?\s*(?:ICE\s+CREAM\s+)?CAKE[S]?[\s\S]{0,50}?(\d+)/i);
      if (cake10Match) {
        cakesCount += parseInt(cake10Match[1]);
      }

      const sheetCakeMatch = pageText.match(/(?:SHEET|QUARTER|HALF)\s*CAKE[S]?[\s\S]{0,50}?(\d+)/i);
      if (sheetCakeMatch) {
        cakesCount += parseInt(sheetCakeMatch[1]);
      }

      if (cakesCount === 0) {
        match = pageText.match(/(?:Ice\s+Cream\s+)?Cakes?\s+(?:Sold)?[:\s]+(\d+)/i);
        if (match) {
          cakesCount = parseInt(match[1]);
        }
      }

      if (cakesCount === 0) {
        const allCakeMatches = pageText.match(/(?:CAKE|CAKES)[\s\S]{0,50}?(\d+)\s*$/gim);
        if (allCakeMatches && allCakeMatches.length > 0) {
          for (const cakeMatch of allCakeMatches) {
            const numMatch = cakeMatch.match(/(\d+)\s*$/);
            if (numMatch) {
              const num = parseInt(numMatch[1]);
              if (num > 0 && num < 100) {
                cakesCount = num;
                break;
              }
            }
          }
        }
      }

      // Extract Cupcake Count with pack size multiplication (can be on any page)
      const pack4Match = pageText.match(/4\s*PK\s*[-\s]*Cupcakes?[\s\S]{0,50}?(\d+)/i);
      if (pack4Match) {
        cupcakesCount += parseInt(pack4Match[1]) * 4;
      }

      const pack6Match = pageText.match(/6\s*PK\s*[-\s]*Cupcakes?[\s\S]{0,50}?(\d+)/i);
      if (pack6Match) {
        cupcakesCount += parseInt(pack6Match[1]) * 6;
      }

      const pack12Match = pageText.match(/12\s*PK\s*[-\s]*Cupcakes?[\s\S]{0,50}?(\d+)/i);
      if (pack12Match) {
        cupcakesCount += parseInt(pack12Match[1]) * 12;
      }

      if (cupcakesCount === 0) {
        match = pageText.match(/(?<![\dPK]\s)Cupcakes?\s+(?:Sold)?[:\s]+(\d+)/i);
        if (match) {
          cupcakesCount = parseInt(match[1]);
        }
      }

      if (cupcakesCount === 0) {
        match = pageText.match(/(?:TREATZZA|CUPCAKE)[S]?\s+(\d+)/i);
        if (match) {
          cupcakesCount = parseInt(match[1]);
        }
      }

      const genericPackMatches = pageText.matchAll(/(\d+)\s*PK\s*[-\s]*Cupcakes?[\s\S]{0,50}?(\d+)/gi);
      for (const packMatch of genericPackMatches) {
        const packSize = parseInt(packMatch[1]);
        const quantity = parseInt(packMatch[2]);
        cupcakesCount += packSize * quantity;
      }

      return {
        totalMedia,
        totalCash,
        creditCards,
        food,
        salesTax,
        customerCount,
        avgGuestCheck,
        nonTaxableSales,
        cakesCount,
        cupcakesCount,
        debugText: pageText
      };
    }, combinedText);

    // Save debug text to file
    const debugPath = path.join(__dirname, 'debug-extracted-text.txt');
    fs.writeFileSync(debugPath, extractionResult.debugText);
    console.log(`   Debug: Extracted text saved to debug-extracted-text.txt\n`);

    await browser.close();

    const reportDate = new Date().toLocaleDateString('en-US', {
      month: '2-digit',
      day: '2-digit',
      year: 'numeric',
      timeZone: 'America/New_York'
    });

    const reportTime = new Date().toLocaleTimeString('en-US', {
      hour: '2-digit',
      minute: '2-digit',
      hour12: true,
      timeZone: 'America/New_York'
    });

    // Display results in terminal
    console.log('╔══════════════════════════════════════════════════════════╗');
    console.log('║                                                          ║');
    console.log('║                  SALES REPORT                         ║');
    console.log('║                                                          ║');
    console.log('╚══════════════════════════════════════════════════════════╝\n');

    console.log(`Store: ${STORE_NAME} (#${STORE_NUMBER})`);
    console.log(`Date: ${reportDate}`);
    console.log(`⏰ Time: ${reportTime}\n`);

    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');

    console.log(`TOTAL MEDIA:           ${extractionResult.totalMedia.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`);
    console.log(`TOTAL CASH:            ${extractionResult.totalCash.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`);
    console.log(`CREDIT CARDS:          ${extractionResult.creditCards.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`);
    console.log(`FOOD:                  ${extractionResult.food.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`);
    console.log(`SALES TAX:             ${extractionResult.salesTax.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`);
    console.log(`CUSTOMER COUNT:        ${extractionResult.customerCount}`);
    console.log(`AVG GUEST CHECK:       ${extractionResult.avgGuestCheck.toFixed(2)}`);
    console.log(`NON-TAXABLE SALES:     ${extractionResult.nonTaxableSales.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`);
    console.log(`CAKES SOLD:            ${extractionResult.cakesCount}`);
    console.log(`CUPCAKES SOLD:         ${extractionResult.cupcakesCount}\n`);

    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');

    if (extractionResult.totalMedia > 0) {
      console.log('✅ Report completed successfully!\n');
    } else {
      console.log('⚠️  Warning: Some fields may be zero. Check debug file.\n');
    }

    // Save to JSON file
    const reportData = {
      store: {
        location: STORE_LOCATION,
        number: STORE_NUMBER,
        name: STORE_NAME
      },
      report: {
        date: reportDate,
        time: reportTime,
        generatedAt: new Date().toISOString()
      },
      sales: {
        totalMedia: extractionResult.totalMedia,
        totalCash: extractionResult.totalCash,
        creditCards: extractionResult.creditCards,
        food: extractionResult.food,
        salesTax: extractionResult.salesTax,
        customerCount: extractionResult.customerCount,
        avgGuestCheck: extractionResult.avgGuestCheck,
        nonTaxableSales: extractionResult.nonTaxableSales,
        cakesCount: extractionResult.cakesCount,
        cupcakesCount: extractionResult.cupcakesCount
      }
    };

    // Save to latest-report.json
    const reportPath = path.join(__dirname, 'latest-report.json');
    fs.writeFileSync(reportPath, JSON.stringify(reportData, null, 2));
    console.log(`Report saved to: ${reportPath}\n`);

    // Also save to history
    const historyDir = path.join(__dirname, 'reports-history');
    if (!fs.existsSync(historyDir)) {
      fs.mkdirSync(historyDir);
    }

    const historyFileName = `report-${new Date().toISOString().split('T')[0]}.json`;
    const historyPath = path.join(historyDir, historyFileName);
    fs.writeFileSync(historyPath, JSON.stringify(reportData, null, 2));
    console.log(`Report archived to: ${historyPath}\n`);

    return reportData;

  } catch (error) {
    console.error('\n❌ Error generating report:', error.message);
    if (browser) await browser.close();
    throw error;
  }
}

// Run the report
fetchReport()
  .then(() => {
    console.log('All done!\n');
    process.exit(0);
  })
  .catch((error) => {
    console.error('\n❌ Report failed:', error.message);
    process.exit(1);
  });
