const http = require('http');
const fs = require('fs');
const path = require('path');
const { exec } = require('child_process');

const PORT = 8080;
const HOST = '0.0.0.0';

// MIME types for different file extensions
const mimeTypes = {
  '.html': 'text/html',
  '.js': 'text/javascript',
  '.json': 'application/json',
  '.css': 'text/css',
  '.png': 'image/png',
  '.jpg': 'image/jpeg',
  '.gif': 'image/gif',
  '.svg': 'image/svg+xml',
  '.ico': 'image/x-icon'
};

const server = http.createServer((req, res) => {
  // Handle print requests
  if (req.url.startsWith('/print?type=')) {
    const printerType = req.url.split('type=')[1];

    console.log(\`\n📄 Print request received: \${printerType.toUpperCase()}\`);
    console.log(\`   Time: \${new Date().toLocaleString('en-US', { timeZone: 'America/New_York' })}\`);

    // Execute the print helper script
    const command = \`node "\${path.join(__dirname, 'print-helper.js')}" \${printerType}\`;

    exec(command, (error, stdout, stderr) => {
      if (error) {
        console.error(\`❌ Print error: \${error.message}\`);
        res.writeHead(500, { 'Content-Type': 'text/plain' });
        res.end(\`Print failed: \${error.message}\`);
        return;
      }

      if (stderr) {
        console.error(\`⚠️  Warning: \${stderr}\`);
      }

      console.log(stdout);
      res.writeHead(200, { 'Content-Type': 'text/plain' });
      res.end('Print job sent successfully!');
    });

    return;
  }

  // Serve static files
  let filePath = '.' + req.url;
  if (filePath === './') {
    filePath = './dashboard-v2.html';
  }

  // Get absolute path
  filePath = path.join(__dirname, filePath);

  // Get file extension
  const extname = String(path.extname(filePath)).toLowerCase();
  const contentType = mimeTypes[extname] || 'application/octet-stream';

  // Read and serve the file
  fs.readFile(filePath, (error, content) => {
    if (error) {
      if (error.code === 'ENOENT') {
        res.writeHead(404, { 'Content-Type': 'text/html' });
        res.end('<h1>404 - File Not Found</h1>', 'utf-8');
      } else {
        res.writeHead(500);
        res.end(\`Server Error: \${error.code}\`);
      }
    } else {
      res.writeHead(200, { 'Content-Type': contentType });
      res.end(content, 'utf-8');
    }
  });
});

server.listen(PORT, HOST, () => {
  console.log('\n╔══════════════════════════════════════════════════════════╗');
  console.log('║                                                          ║');
  console.log('║          D&K DQ - PRINT-ENABLED DASHBOARD SERVER         ║');
  console.log('║                                                          ║');
  console.log('╚══════════════════════════════════════════════════════════╝\n');
  console.log(\`✓ Server running at http://localhost:\${PORT}/\`);
  console.log(\`✓ Dashboard URL: http://localhost:\${PORT}/dashboard.html\`);
  console.log(\`✓ Print endpoint: http://localhost:\${PORT}/print?type=[printer]\`);
  console.log(\`\n📄 Supported printers:\`);
  console.log(\`   • COM1-COM5 (Serial/USB): /print?type=com1\`);
  console.log(\`   • Network (192.168.192.123:9100): /print?type=network\`);
  console.log(\`\n⚠️  Press Ctrl+C to stop the server\n\`);
});

// Handle server errors
server.on('error', (error) => {
  if (error.code === 'EADDRINUSE') {
    console.error(\`\n❌ ERROR: Port \${PORT} is already in use!\`);
    console.error('   Another program is using this port.');
    console.error('   Close the other program and try again.\n');
    process.exit(1);
  } else {
    console.error(\`\n❌ Server error: \${error.message}\n\`);
    process.exit(1);
  }
});

// Graceful shutdown
process.on('SIGINT', () => {
  console.log('\n\n✓ Server stopped gracefully.\n');
  server.close(() => {
    process.exit(0);
  });
});
