const fs = require('fs');
const net = require('net');
const path = require('path');

// Get printer type from command line arguments
const args = process.argv.slice(2);
const printerType = args[0]; // 'com1', 'com2', 'com3', 'com4', 'com5', or 'network'

console.log('\n╔══════════════════════════════════════════════════════════╗');
console.log('║                                                          ║');
console.log('║          D&K DQ - AUTOMATIC RECEIPT PRINTER              ║');
console.log('║              Epson TM-T88VI Thermal Printer              ║');
console.log('║                                                          ║');
console.log('╚══════════════════════════════════════════════════════════╝\n');

console.log(`Printer Type: ${printerType?.toUpperCase() || 'NONE'}`);
console.log(`Time: ${new Date().toLocaleString('en-US', { timeZone: 'America/New_York' })}\n`);

// Read the latest report data
const reportPath = path.join(__dirname, 'latest-report.json');
let reportData;

try {
  reportData = JSON.parse(fs.readFileSync(reportPath, 'utf8'));
} catch (error) {
  console.error('❌ ERROR: Could not read report file!');
  console.error('   Make sure to run the report first.\n');
  process.exit(1);
}

// ESC/POS commands for Epson TM-T88VI
const ESC = '\x1B';
const GS = '\x1D';

// Initialize printer
const INIT = ESC + '@';

// Text formatting
const BOLD_ON = ESC + 'E' + '\x01';
const BOLD_OFF = ESC + 'E' + '\x00';
const CENTER = ESC + 'a' + '\x01';
const LEFT = ESC + 'a' + '\x00';
const RIGHT = ESC + 'a' + '\x02';

// Font sizes
const NORMAL = GS + '!' + '\x00';
const DOUBLE_HEIGHT = GS + '!' + '\x01';
const DOUBLE_WIDTH = GS + '!' + '\x10';
const DOUBLE_SIZE = GS + '!' + '\x11';

// Line feed and cut
const LF = '\n';
const CUT = GS + 'V' + '\x42' + '\x00'; // Partial cut

// Build receipt content in ESC/POS format
function buildReceiptESCPOS(data) {
  const sales = data.sales || {};
  const store = data.store || {};
  const report = data.report || {};

  let receipt = '';

  // Initialize printer
  receipt += INIT;

  // Header - Centered, Bold, Double Size
  receipt += CENTER + DOUBLE_SIZE + BOLD_ON;
  receipt += 'D&K DAIRY QUEEN' + LF;
  receipt += NORMAL + BOLD_OFF;

  // Store name and number
  receipt += BOLD_ON + DOUBLE_WIDTH;
  receipt += (store.name || 'Unknown') + ' #' + (store.number || '00000') + LF;
  receipt += NORMAL + BOLD_OFF;

  // Date and time
  receipt += (report.date || '') + ' ' + (report.time || '') + LF;

  // Separator line
  receipt += LEFT + '--------------------------------' + LF;

  // Sales data - Left aligned
  receipt += LF;
  receipt += BOLD_ON + 'TOTAL MEDIA:' + BOLD_OFF + LF;
  receipt += '  $' + (sales.totalMedia || 0).toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + LF;
  receipt += LF;

  receipt += BOLD_ON + 'TOTAL CASH:' + BOLD_OFF + LF;
  receipt += '  $' + (sales.totalCash || 0).toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + LF;
  receipt += LF;

  receipt += BOLD_ON + 'CREDIT CARDS:' + BOLD_OFF + LF;
  receipt += '  $' + (sales.creditCards || 0).toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + LF;
  receipt += LF;

  receipt += BOLD_ON + 'FOOD:' + BOLD_OFF + LF;
  receipt += '  $' + (sales.food || 0).toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + LF;
  receipt += LF;

  receipt += BOLD_ON + 'SALES TAX:' + BOLD_OFF + LF;
  receipt += '  $' + (sales.salesTax || 0).toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + LF;
  receipt += LF;

  receipt += BOLD_ON + 'CUSTOMER COUNT:' + BOLD_OFF + LF;
  receipt += '  ' + (sales.customerCount || 0) + ' guests' + LF;
  receipt += LF;

  receipt += BOLD_ON + 'AVG GUEST CHECK:' + BOLD_OFF + LF;
  receipt += '  $' + (sales.avgGuestCheck || 0).toFixed(2) + LF;
  receipt += LF;

  receipt += BOLD_ON + 'NON-TAXABLE SALES:' + BOLD_OFF + LF;
  receipt += '  $' + (sales.nonTaxableSales || 0).toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + LF;

  // Separator
  receipt += LF + '--------------------------------' + LF + LF;

  // Product counts
  receipt += BOLD_ON + 'CAKES SOLD:' + BOLD_OFF + LF;
  receipt += '  ' + (sales.cakesCount || 0) + ' cakes' + LF;
  receipt += LF;

  receipt += BOLD_ON + 'CUPCAKES SOLD:' + BOLD_OFF + LF;
  receipt += '  ' + (sales.cupcakesCount || 0) + ' cupcakes' + LF;

  // Footer
  receipt += LF + '--------------------------------' + LF;
  receipt += CENTER;
  receipt += 'AUTOMATED REPORT' + LF;
  receipt += 'PARBRINK POS SYSTEM' + LF;
  receipt += LF + LF + LF;

  // Cut paper
  receipt += CUT;

  return receipt;
}

// Print to COM port (Serial/USB)
function printToCOM(portNumber, receiptData) {
  const SerialPort = require('serialport');
  const port = new SerialPort(`COM${portNumber}`, {
    baudRate: 9600, // Standard for Epson TM-T88VI
    dataBits: 8,
    parity: 'none',
    stopBits: 1,
    flowControl: false
  }, (err) => {
    if (err) {
      console.error(`❌ ERROR: Could not open COM${portNumber}`);
      console.error(`   Error: ${err.message}`);
      console.error(`\n   Troubleshooting:`);
      console.error(`   1. Check if printer is connected to COM${portNumber}`);
      console.error(`   2. Make sure no other program is using the port`);
      console.error(`   3. Try a different COM port`);
      console.error(`   4. Check Device Manager for COM port number\n`);
      process.exit(1);
    }

    console.log(`✓ Successfully opened COM${portNumber}`);
    console.log(`  Baud Rate: 9600`);
    console.log(`  Data Bits: 8, Parity: None, Stop Bits: 1\n`);

    // Send data to printer
    console.log('Sending receipt data to printer...');
    port.write(Buffer.from(receiptData, 'binary'), (err) => {
      if (err) {
        console.error('❌ ERROR: Failed to write to COM port');
        console.error(`   ${err.message}\n`);
        port.close();
        process.exit(1);
      }

      console.log('✓ Receipt data sent successfully!\n');

      // Wait for data to be transmitted, then close
      port.drain(() => {
        console.log('✓ Print job completed!');
        console.log('✓ Receipt should be printing now...\n');
        port.close();
        process.exit(0);
      });
    });
  });
}

// Print to network printer (IP address)
function printToNetwork(ipAddress, port, receiptData) {
  console.log(`Connecting to network printer:`);
  console.log(`  IP Address: ${ipAddress}`);
  console.log(`  Port: ${port}`);
  console.log(`  Protocol: Raw TCP/IP Socket\n`);

  const client = new net.Socket();

  // Set timeout for connection
  client.setTimeout(5000);

  client.connect(port, ipAddress, () => {
    console.log('✓ Connected to network printer!\n');
    console.log('Sending receipt data...');

    // Send data to printer
    client.write(Buffer.from(receiptData, 'binary'), () => {
      console.log('✓ Receipt data sent successfully!\n');
    });

    // Wait a moment for data to transmit, then close
    setTimeout(() => {
      client.destroy();
      console.log('✓ Print job completed!');
      console.log('✓ Receipt should be printing now...\n');
      process.exit(0);
    }, 1000);
  });

  client.on('error', (err) => {
    console.error('❌ ERROR: Could not connect to network printer');
    console.error(`   ${err.message}\n`);
    console.error('   Troubleshooting:');
    console.error(`   1. Check if printer IP is ${ipAddress}`);
    console.error(`   2. Make sure printer is powered on and connected to network`);
    console.error(`   3. Ping the printer: ping ${ipAddress}`);
    console.error(`   4. Check firewall settings`);
    console.error(`   5. Verify port ${port} is open on the printer\n`);
    process.exit(1);
  });

  client.on('timeout', () => {
    console.error('❌ ERROR: Connection timeout');
    console.error('   Printer did not respond within 5 seconds.\n');
    client.destroy();
    process.exit(1);
  });
}

// Main execution
const receiptData = buildReceiptESCPOS(reportData);

switch (printerType) {
  case 'com1':
    printToCOM(1, receiptData);
    break;
  case 'com2':
    printToCOM(2, receiptData);
    break;
  case 'com3':
    printToCOM(3, receiptData);
    break;
  case 'com4':
    printToCOM(4, receiptData);
    break;
  case 'com5':
    printToCOM(5, receiptData);
    break;
  case 'network':
    printToNetwork('192.168.192.123', 9100, receiptData);
    break;
  default:
    console.error('❌ ERROR: Invalid printer type!');
    console.error('   Usage: node print-helper.js [com1|com2|com3|com4|com5|network]\n');
    console.error('   Examples:');
    console.error('     node print-helper.js com1     - Print to COM1');
    console.error('     node print-helper.js network  - Print to network printer\n');
    process.exit(1);
}
